/***************************************************************************
 *************************** Expression Evaluator **************************
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <setjmp.h>

#include "evaldefs.h"

#define ERR(n) {ev_ERROR=n; ev_ERPOS=expression-ev_ERANC-1; longjmp(jb,1);}

#ifndef M_PI
#define M_PI	3.14159265358979323846
#endif

#ifndef M_E
#define M_E	2.71828182845904523536
#endif

/* Variables dealing with errors */
int  ev_ERROR;			/* The error number */
int  ev_ERPOS;			/* The offset from the start of the expression */
char *ev_ERANC; 		/* Used to calculate ERPOS */

/* Pointer to external user-defined variables */
VARIABLE *Vars=NULL;

/* Constants used in expression
   (name, value) */
VARIABLE Consts[]=
{
   {"pi", M_PI},
   {"e",  M_E},
   {0}
};

/* Prototypes for custom math functions */
TYPE deg(TYPE x);
TYPE rad(TYPE x);

/* Math functions recognized in expression
   (name, parameters, pointer) */
FUNCTION Funcs[]=
{
   {"sin",    1, sin},
   {"cos",    1, cos},
   {"tg",     1, tan},
   {"arcsin", 1, asin},
   {"arccos", 1, acos},
   {"arctg",  1, atan},
   {"sh",     1, sinh},
   {"ch",     1, cosh},
   {"th",     1, tanh},
   {"exp",    1, exp},
   {"ln",     1, log},
   {"lg",     1, log10},
   {"sqrt",   1, sqrt},
   {"floor",  1, floor},
   {"ceil",   1, ceil},
   {"abs",    1, fabs},
   {"deg",    1, deg},
   {"rad",    1, rad},
   {0}
};

char*	expression;	     /* Pointer to the user's expression */
char	token[TOKLEN+1]; /* Holds the current token */
int		type;		     /* Type of the current token */
jmp_buf 	jb;		     /* jmp_buf for errors */

/* Internal routines */
void Parse(void);
void Level1(TYPE *r);
void Level2(TYPE *r);
void Level3(TYPE *r);
void Level4(TYPE *r);
void Level5(TYPE *r);

/* Converts x radians to degrees. */
double deg(double x)
{
   return (x*180.0/M_PI);
}
/* Converts x degrees to radians. */
double rad(double x)
{
   return (x*M_PI/180.0);
}

/* Gets the value of variable */
int GetValue(char *name,TYPE* value)
{
   int i;
   /* First search in Consts */
   for (i=0;*Consts[i].name;i++)
	 if ((*Consts[i].name)&&(!strcmp(name,Consts[i].name)))
	 {
	*value=Consts[i].value;
	return (1);
	 }
   /* Then search in Vars */
   if (Vars)
   {
	  for (i=0;*Vars[i].name;i++)
	if ((*Vars[i].name)&&(!strcmp(name,Vars[i].name)))
	{
	   *value=Vars[i].value;
	   return (1);
	}
   }
   return (0);
}

/* Grab the next token from the expression */
void Parse(void)
{
   char* t;

   type=0;
   t=token;

   while (isspace(*expression)) expression++;

   if (isdelim(*expression))
   {
	  type=DELIM;
	  *t++=*expression++;
   }
   else if (isnumer(*expression))
   {
	  type=NUMER;
	  while (isnumer(*expression)) *t++=*expression++;
   }
   else if (isalpha(*expression))
   {
	  type=NAME;
	  while (isalpha(*expression)) *t++=*expression++;
	  token[VARLEN]=0;
   }
   else if (*expression)
   {
	  *t++=*expression++;
	  *t=0;
	  ERR(ev_SYNTAX);
   }
   *t=0;
   while (isspace(*expression)) expression++;
}

/* This function handles any addition and subtraction operations. */
void Level1(TYPE *r)
{
   TYPE t=0;
   char op;

   Level2(r);
   while (((op=*token)=='+')||(op=='-'))
   {
	  Parse();
	  Level2(&t);
	  if (op=='+') *r=*r+t;
	  else if (op=='-') *r=*r-t;
   }
}

/* This function handles any multiplication, division, or modulo. */
void Level2(TYPE *r)
{
   TYPE t;
   char op;

   Level3(r);
   while (((op=*token)=='*')||(op=='/')||(op=='%'))
   {
      Parse();
      Level3(&t);
      if (op=='*') *r=*r*t;
      else if (op=='/')
      {
	 if (t==0) ERR(ev_DIVZERO);
	 *r=*r/t;
      }
      else if (op=='%')
      {
	 if (t==0) ERR(ev_DIVZERO);
	 *r=fmod(*r,t);
      }
   }
}

/* This function handles any "to the power of" operations. */
void Level3(TYPE *r)
{
   TYPE t;

   Level4(r);
   if (*token=='^')
   {
      Parse();
      Level4(&t);
      *r=pow(*r,t);
   }
}

/* This function handles any unary + or - signs. */
void Level4(TYPE *r)
{
   char op=0;

   if (*token=='+'||*token=='-')
   {
	  op=*token;
	  Parse();
   }
   Level5(r);
   if (op=='-') *r=-*r;
}

/* This function handles any literal numbers, variables, or functions. */
void Level5(TYPE *r)
{
   int	i;
   int	n;
   TYPE a[MAXARG];

   if (*token=='(')
   {
	  Parse();
	  if (*token==')') ERR(ev_NOARG);
	  Level1(r);
	  if (*token!=')') ERR(ev_UNBALAN);
	  Parse();
   }
   else if (type==NUMER)
   {
	  *r=(TYPE)atof(token);
	  Parse();
   }
   else if (type==NAME)
   {
	  if (*expression=='(')
	  {
	 for (i=0;*Funcs[i].name;i++)
	   if (!strcmp(token,Funcs[i].name))
	   {
		  Parse();
		  n=0;
		  do
		  {
		 Parse();
		 if ((*token==')')||(*token==',')) ERR(ev_NOARG);
		 a[n]=0;
		 Level1(&a[n]);
		 n++;
		  }  while ((n<MAXARG)&&(*token==','));
		  if (*token!=')') ERR(ev_UNBALAN);
		  Parse();
		  if (n!=Funcs[i].args)
		  {
		 strcpy(token,Funcs[i].name);
		 ERR(ev_NUMARGS);
		  }
		  *r=Funcs[i].func(a[0],a[1],a[2],a[3]);
		  return;
	   }
	 if (!*Funcs[i].name) ERR(ev_BADFUNC);
      }
      else if (!GetValue(token,r)) ERR(ev_UNKNOWN);
      Parse();
   }
   else ERR(ev_SYNTAX);
}

/* This function evaluates an expression. */
int Evaluate(char *e,VARIABLE *v,TYPE* result)
{
   Vars=v;
   if (setjmp(jb)) return(ev_ERROR);
   expression=e;
   ev_ERANC=e;
   strlwr(expression);
   *result=0;
   Parse();
   if (!*token) ERR(ev_EMPTY);
   Level1(result);
   return(ev_OK);
}

/* Math error hndler */
int _matherr(struct exception *e)
  {
   ERR(ev_MATH);
   return 0;
  }